package org.opensha.sra.vulnerability.models;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.opensha.commons.data.function.ArbitrarilyDiscretizedFunc;
import org.opensha.commons.data.function.AbstractDiscretizedFunc;
import org.apache.commons.math3.distribution.LogNormalDistribution;
import org.opensha.commons.util.StatUtil;
import org.opensha.sra.vulnerability.AbstractVulnerability;

/**
 * This is a simple, perhaps temporary class for instantiating AbstractVulerability objects
 * from files.
 * 
 * @author kevin
 *
 */
public class SimpleVulnerability extends AbstractVulnerability implements Serializable {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 2L;
	
	private ArrayList<Double> mdfVals;
	private ArrayList<Double> covVals;
	private ArrayList<Double> im_levels;
	
	private AbstractDiscretizedFunc vulnFunc;
	private AbstractDiscretizedFunc covFunc;
	
    private double[][] DEM;
    private double[] DEM_DF_vals;
	
	private static double[] toPrimDoubleArray(List<Double> vals) {
		double dvals[] = new double[vals.size()];
		
		for (int i=0; i<vals.size(); i++) {
			dvals[i] = vals.get(i);
		}
		return dvals;
	}
	
	public SimpleVulnerability(String name, String shortName, String im_type, ArrayList<Double> im_levels,
			ArrayList<Double> mdfVals, ArrayList<Double> covVals) {
		super(name, shortName, im_type, toPrimDoubleArray(im_levels));
		
		if (im_levels.size() != mdfVals.size())
			throw new IllegalArgumentException("im_levels must be the same size as mdfVals");
		
		if (im_levels.size() != covVals.size())
			throw new IllegalArgumentException("im_levels must be the same size as covVals");
		
		// damage factors for damage exceedance matrix
	    this.DEM_DF_vals = getDamageFactorsForDEM();
	    
	    // building the damage exceedance matrix
	    this.DEM = buildDEMMatrix(toPrimDoubleArray(im_levels), toPrimDoubleArray(mdfVals), toPrimDoubleArray(covVals), this.DEM_DF_vals);
		
		this.im_levels = im_levels;
		this.mdfVals = mdfVals;
		this.covVals = covVals;
		vulnFunc = new ArbitrarilyDiscretizedFunc();
		covFunc = new ArbitrarilyDiscretizedFunc();
		for (int i=0; i<im_levels.size(); i++) {
			vulnFunc.set(im_levels.get(i), mdfVals.get(i));
			covFunc.set(im_levels.get(i), covVals.get(i));
		}
		String funcName = name;
		if (shortName != null && !shortName.equals(name))
			funcName += " ("+shortName+")";
		vulnFunc.setName(funcName);
	}

    /**
     * Internal helper class that wraps Apache Commons Math's LogNormalDistribution.
     *
     * This class converts a mean value (MDF) and a coefficient of variation (COV)
     * into parameters of the underlying lognormal distribution:
     *
     *     sigma = sqrt( ln(1 + COV^2) )
     *     mu = ln(mean) - (sigma^2)/2
     *
     */
    private static class LogNormalHelper {

        /** LogNormalDistribution instance from Apache Commons Math. */
        private final LogNormalDistribution dist;

        /**
         * Constructs a lognormal distribution from the specified mean and COV.
         *
         * @param mean  mean damage factor (DF[i]) at a given IML
         * @param cov   coefficient of variation of damage factor (COV[i])
         */
        private LogNormalHelper(double mean, double cov) {
            // Convert mean & COV into lognormal parameters
            double sigma = Math.sqrt(Math.log(1.0 + cov * cov));
            double mu = Math.log(mean) - 0.5 * sigma * sigma;

            // Construct a numerically stable lognormal distribution
            this.dist = new LogNormalDistribution(mu, sigma);
        }

        /**
         * Computes the cumulative distribution function (CDF) value for the
         * lognormal distribution at the specified damage factor.
         *
         * @param x damage factor threshold
         * @return P(Damage <= x) for the underlying lognormal
         */
        public double cdf(double x) {
            return dist.cumulativeProbability(x);
        }
    }
    
    /**
     * Generates a geometrically spaced array of values from start to end.
     *
     * @param start the starting value
     * @param end the maximum value
     * @param ratio the geometric ratio between consecutive values
     * @return array of values
     */
    private static double[] generateDamageFactorsForDEM(double start, double end, double ratio) {
    	List<Double> values = new ArrayList<>();
        for (double x = start; x <= end; x *= ratio) {
            // Add values spaced geometrically by ratio
            values.add(x);
        }
        return toPrimDoubleArray(values);
    }
    
    
    /**
     * Generates an array of damage factor thresholds for building the DEM matrix.
     * 
     * The start, increment, and end points (1e-6, 1.0, 1.2) are consistent with those
     * hardcoded in the Curee vulnerabilities
     *
     * @return array of damage factors
     */
    public static double[] getDamageFactorsForDEM() {
        return generateDamageFactorsForDEM(1e-6, 1.0, 1.2);
    }
    
    /**
     * Builds the Damage Exceedance Matrix (DEM) for a vulnerability.
     *
     * Each column corresponds to an intensity measure level (IML).
     * Each row corresponds to a damage-factor threshold (DEM_DF).
     *
     * Element DEM[j][i] = P(Damage > DEM_DFs[j] | IML = imls[i])
     *
     * The damage factor at each intensity measure level is assumed to follow a lognormal
     * distribution defined by DF[i] (mean) and COVDF[i] (coefficient of variation).
     *
     * Special case:
     * - If COVDF[i] == 0, the damage is treated as deterministic (delta function).
     *   In this case, DEM[j][i] = 1.0 if DF[i] >= DEM_DFs[j], else 0.0.
     *
     * @param imls    array of intensity measure levels (length = nIML)
     * @param DF      array of mean damage factor values at each IML (length = nIML)
     * @param COVDF   array of coefficient of variation of DF at each IML (length = nIML)
     * @param DEM_DFs array of damage-factor thresholds (length = nThresholds)
     * @return        DEM matrix [nThresholds][nIML] where DEM[j][i] = P(Damage > DEM_DFs[j] | IML[i])
     */
    private static double[][] buildDEMMatrix(double[] imls,
                                             double[] DF,
                                             double[] COVDF,
                                             double[] DEM_DFs) {

        double[][] DEM = new double[DEM_DFs.length][imls.length];

        // Loop over intensity measure levels (columns)
        for (int i = 0; i < imls.length; i++) {

            // Handle deterministic case (COV = 0)
            if (COVDF[i] == 0) {
                for (int j = 0; j < DEM_DFs.length; j++) {
                    // Step function: 1 if DF >= threshold, 0 if DF < threshold
                    DEM[j][i] = (DF[i] >= DEM_DFs[j] ? 1.0 : 0.0);
                }
                continue; // Skip lognormal calculation
            }

            // Lognormal distribution for probabilistic damage
            LogNormalHelper logNorm = new LogNormalHelper(DF[i], COVDF[i]);

            // Compute exceedance probabilities for each damage threshold
            for (int j = 0; j < DEM_DFs.length; j++) {
                DEM[j][i] = 1.0 - logNorm.cdf(DEM_DFs[j]);
            }
        }

        return DEM;
    }
    

	@Override
	public double[] getDEMDFVals() {
		return DEM_DF_vals;
	}

	@Override
	public double[][] getDEMMatrix() {
		return DEM;
	}

	@Override
	public double getDF(double IML) {
		// TODO Auto-generated method stub
		return 0;
	}

	@Override
	public ArrayList<double[]> getDFTable() {
		// TODO Auto-generated method stub
		return null;
	}
	
	@Override
	public AbstractDiscretizedFunc getVulnerabilityFunc() {
		return vulnFunc;
	}
	
	@Override
	public String toString() {
		String str = "Simple Vulnurabilty Function: " + this.getName() + " (" + this.getShortName() + ")" + "\n";
		str += "IM Type: " + this.getIMT() + "\n";
		str += "IML\tMDF\tCOV\n";
		for (int i=0; i<im_levels.size(); i++) {
			str += im_levels.get(i) + "\t" + mdfVals.get(i) + "\t" + covVals.get(i) + "\n";
		}
		str += "total vals: " + im_levels.size();
		
		return str;
	}

	@Override
	public double getMeanDamageAtExceedProb(double iml, double prob) {
		// inverse of the cumulative standard normal distribution evaluated at p (like normsinv in Microsoft Excel)
		double invCumStdNormDist = StatUtil.getInvCDF(1-prob, true);
		// coefficient of variation of damage factor at iml
		double d = covFunc.getInterpolatedY(iml);
		// mean damage factor at iml
		double y = this.getMeanDamageFactor(iml);
		// logarithmic standard deviation of damage factor at iml
		double beta = Math.sqrt(Math.log(1 + d*d));
		// median damage factor at iml
		double yhat = y / Math.sqrt(1 + d*d);
		// damage factor at exceedance probability p, given iml
		double yp = yhat * Math.exp(beta*invCumStdNormDist);
		
//		System.out.println("yp: " + yp);
		return yp;
	}
	
	public AbstractDiscretizedFunc getCOVFunction() {
		return covFunc;
	}

	@Override
	public double getMeanDamageFactor(double iml) {
		return vulnFunc.getInterpolatedY(iml);
	}

}
