package org.opensha.sha.faultSurface;

import java.util.ArrayList;
import java.util.Iterator;

import org.opensha.commons.exceptions.FaultException;
import org.opensha.commons.geo.LocationList;
import org.opensha.commons.geo.LocationVector;
import org.opensha.commons.geo.Location;
import org.opensha.commons.geo.LocationUtils;
import org.opensha.commons.geo.Region;





/**
 * <b>Title:</b> FrankelGriddedSurface.  This creates an
 * EvenlyDiscretizedSurface using the scheme defined by Art Frankel in his
 * Fortran code for the 1996 USGS hazard maps.  Grid points are projected down
 * dip at an angle perpendicular to the azimuth of each segment.<br>
 * <b>Description:</b> <br>
 * @author Steven W. Rock
 * @version 1.0
 */

public class FrankelGriddedSurface extends EvenlyGriddedSurfFromSimpleFaultData {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	protected final static String C = "FrankelGriddedSurface";
	protected final static boolean D = false;

	protected final static double PI_RADIANS = Math.PI / 180;
	protected final static String ERR = " is null, unable to process.";


	/**
	 * This applies the  grid spacing exactly as given (trimming any remainder from the ends),
	 */
	public FrankelGriddedSurface( SimpleFaultData simpleFaultData,
			double gridSpacing)
	throws FaultException {

		super(simpleFaultData, gridSpacing);
		createEvenlyGriddedSurface();
	}

	/**
	 * This applies the  grid spacing exactly as given (trimming any remainder from the ends),
	 */
	public FrankelGriddedSurface( FaultTrace faultTrace,
			double aveDip,
			double upperSeismogenicDepth,
			double lowerSeismogenicDepth,
			double gridSpacing )
	throws FaultException {

		super(faultTrace, aveDip, upperSeismogenicDepth, lowerSeismogenicDepth, gridSpacing);
		createEvenlyGriddedSurface();
	}

	/**
	 * Stitch Together the fault sections. It assumes:
	 * 1. Sections are in correct order
	 * 2. Distance between end points of section in correct order is less than the distance to opposite end of section
	 * Upper seismogenic depth, sip aand lower seimogenic depth are area weighted.
	 * 
	 * @param simpleFaultData
	 * @param gridSpacing
	 * @throws FaultException
	 */
	public FrankelGriddedSurface(ArrayList<SimpleFaultData> simpleFaultData,
			double gridSpacing) throws FaultException {
		super(simpleFaultData, gridSpacing);
		createEvenlyGriddedSurface();
	}
	
	/**
	 * This will fit the length & DDW of the fault exactly (no trimming) by adjusting the grid spacing
	 * to just below the grid spacings given
	 * @param simpleFaultData
	 * @param maxGridSpacingAlong
	 * @param maxGridSpacingDown
	 * @throws FaultException
	 */
    public FrankelGriddedSurface(SimpleFaultData simpleFaultData, double maxGridSpacingAlong, double maxGridSpacingDown) throws FaultException {
        super(simpleFaultData, maxGridSpacingAlong, maxGridSpacingDown);
        createEvenlyGriddedSurface();
      }

    /**
     * Default constructor - private and only used for cloning purposes
     */
    private FrankelGriddedSurface() {
    	
    }

	/**
	 * Creates the Frankel Gridded Surface from the Simple Fault Data
	 * @throws FaultException
	 */
	private void createEvenlyGriddedSurface() throws FaultException {

		String S = C + ": createEvenlyGriddedSurface():";
		if( D ) System.out.println(S + "Starting");

		assertValidData();

		final int numSegments = faultTrace.getNumLocations() - 1;
		final double avDipRadians = aveDip * PI_RADIANS;
		final double gridSpacingCosAveDipRadians = gridSpacingDown * Math.cos( avDipRadians );
		final double gridSpacingSinAveDipRadians = gridSpacingDown * Math.sin( avDipRadians );

		double[] segmentLenth = new double[numSegments];
		double[] segmentAzimuth = new double[numSegments];
		double[] segmentCumLenth = new double[numSegments];

		double cumDistance = 0;
		int i = 0;

		// Iterate over each Location in Fault Trace
		// Calculate distance, cumulativeDistance and azimuth for
		// each segment
		Iterator<Location> it = faultTrace.iterator();
		Location firstLoc = it.next();
		Location lastLoc = firstLoc;
		Location loc = null;
		LocationVector dir = null;
		while( it.hasNext() ){

			loc = it.next();
			dir = LocationUtils.vector(lastLoc, loc);

			double azimuth = dir.getAzimuth();
			double distance = dir.getHorzDistance();
			cumDistance += distance;

			segmentLenth[i] = distance;
			segmentAzimuth[i] = azimuth;
			segmentCumLenth[i] = cumDistance;

			i++;
			lastLoc = loc;

		}

		// Calculate down dip width
		double downDipWidth = (lowerSeismogenicDepth-upperSeismogenicDepth)/Math.sin( avDipRadians );

		// Calculate the number of rows and columns
		int rows = 1 + Math.round((float) (downDipWidth/gridSpacingDown));
		int cols = 1 + Math.round((float) (segmentCumLenth[numSegments - 1] / gridSpacingAlong));


		if(D) System.out.println("numLocs: = " + faultTrace.getNumLocations());
		if(D) System.out.println("numSegments: = " + numSegments);
		if(D) System.out.println("firstLoc: = " + firstLoc);
		if(D) System.out.println("lastLoc(): = " + lastLoc);
		if(D) System.out.println("downDipWidth: = " + downDipWidth);
		if(D) System.out.println("totTraceLength: = " + segmentCumLenth[ numSegments - 1]);
		if(D) System.out.println("numRows: = " + rows);
		if(D) System.out.println("numCols: = " + cols);


		// Create GriddedSurface
		int segmentNumber, ith_row, ith_col = 0;
		double distanceAlong, distance, hDistance, vDistance;

		//location object
		Location location1;
		//initialize the num of Rows and Cols for the container2d object that holds
		setNumRowsAndNumCols(rows,cols);


		// Loop over each column - ith_col is ith grid step along the fault trace
		if( D ) System.out.println(S + "Iterating over columns up to " + cols );
		while( ith_col < cols ){

			if( D ) System.out.println(S + "ith_col = " + ith_col);

			// calculate distance from column number and grid spacing
			distanceAlong = ith_col * gridSpacingAlong;
			if( D ) System.out.println(S + "distanceAlongFault = " + distanceAlong);

			// Determine which segment distanceAlong is in
			segmentNumber = 1;
			while( segmentNumber <= numSegments && distanceAlong > segmentCumLenth[ segmentNumber - 1] ){
				segmentNumber++;
			}
			// put back in last segment if grid point has just barely stepped off the end of the trace
			if( segmentNumber == numSegments+1) segmentNumber--;

			if( D ) System.out.println(S + "segmentNumber " + segmentNumber );

			// Calculate the distance from the last segment point
			if ( segmentNumber > 1 ) distance = distanceAlong - segmentCumLenth[ segmentNumber - 2 ];
			else distance = distanceAlong;
			if( D ) System.out.println(S + "distanceFromLastSegPt " + distance );

			// Calculate the grid location along fault trace and put into grid
			location1 = faultTrace.get( segmentNumber - 1 );
			//            dir = new LocationVector(0, distance, segmentAzimuth[ segmentNumber - 1 ], 0);
			dir = new LocationVector(segmentAzimuth[ segmentNumber - 1 ],  distance, 0);

			// location on the trace
			Location traceLocation = LocationUtils.location( location1, dir  );

			// get location at the top of the fault surface
			Location topLocation;
			if(traceLocation.getDepth() < upperSeismogenicDepth) {
				//                vDistance = traceLocation.getDepth()-upperSeismogenicDepth;
				vDistance = upperSeismogenicDepth - traceLocation.getDepth();
				hDistance = vDistance / Math.tan( avDipRadians );
				//                dir = new LocationVector(vDistance, hDistance, segmentAzimuth[ segmentNumber - 1 ]+90, 0);
				dir = new LocationVector(segmentAzimuth[ segmentNumber - 1 ]+90, hDistance, vDistance);
				topLocation = LocationUtils.location( traceLocation, dir );
			}
			else
				topLocation = traceLocation;

			set(0, ith_col, topLocation.clone());
			if( D ) System.out.println(S + "(x,y) topLocation = (0, " + ith_col + ") " + topLocation );

			// Loop over each row - calculating location at depth along the fault trace
			ith_row = 1;
			while(ith_row < rows){

				if( D ) System.out.println(S + "ith_row = " + ith_row);

				// Calculate location at depth and put into grid
				hDistance = ith_row * gridSpacingCosAveDipRadians;
				vDistance = ith_row * gridSpacingSinAveDipRadians;
				//                vDistance = -ith_row * gridSpacingSinAveDipRadians;

				//                dir = new LocationVector(vDistance, hDistance, segmentAzimuth[ segmentNumber - 1 ]+90, 0);
				dir = new LocationVector(segmentAzimuth[ segmentNumber - 1 ]+90, hDistance, vDistance);

				Location depthLocation = LocationUtils.location( topLocation, dir );
				set(ith_row, ith_col, depthLocation.clone());
				if( D ) System.out.println(S + "(x,y) depthLocation = (" + ith_row + ", " + ith_col + ") " + depthLocation );

				ith_row++;
			}
			ith_col++;
		}

		if( D ) System.out.println(S + "Ending");

	}
	
	@Override
	public double getAveDipDirection() {
		return faultTrace.getDipDirection();
	}
	
	/**
	 * This returns a deep copy of this FrankelGriddedSurface
	 * @return
	 */
	public FrankelGriddedSurface deepCopy() {
		return deepCopyOverrideDepth(Double.NaN);
	}
	
	/**
	 * This returns a deep copy of this FrankelGriddedSurface with all depths set to the given depth (if depth is NaN, the original
	 * depth will be preserved). This is useful for background seismicity sources that create copies of Frankel surfaces with only
	 * the depths changed.
	 * @param depth new depth value to override all depths in cloned surface, or NaN to preserve original depths
	 * @return
	 */
	public FrankelGriddedSurface deepCopyOverrideDepth(double depth) {
		FrankelGriddedSurface surf = new FrankelGriddedSurface();
		
		surf.set(faultTrace.clone(), aveDip, upperSeismogenicDepth, lowerSeismogenicDepth, gridSpacingAlong, gridSpacingDown);
		surf.setName(getName());
		surf.setNumRowsAndNumCols(getNumRows(), getNumCols());
		
		for (int row=0; row<getNumRows(); row++) {
			for (int col=0; col<getNumCols(); col++) {
				Location loc = get(row, col);
				if (Double.isNaN(depth))
					surf.set(row, col, loc); // don't have to clone location as it is immutable
				else 
					surf.set(row, col, new Location(loc.getLatitude(), loc.getLongitude(), depth));
			}
		}
		
		return surf;
	}

	public static void main(String args[]) {

		// for N-S strike and E dip, this setup showed that prior to fixing
		// LocationUtils.getLocation() the grid of the fault actually
		// starts to the left of the trace, rather than to the right.
		double aveDip = 30;
		double upperSeismogenicDepth = 5;
		double lowerSeismogenicDepth = 15;
		double gridSpacing=5;
		FaultTrace faultTrace = new FaultTrace("Test");
		faultTrace.add(new Location(20.0, -120, 0));
		faultTrace.add(new Location(20.2, -120, 0));
		FrankelGriddedSurface griddedSurface = new FrankelGriddedSurface(faultTrace, aveDip,
				upperSeismogenicDepth, lowerSeismogenicDepth, gridSpacing);
		System.out.println("******Fault Trace*********");
		System.out.println(faultTrace);
		Iterator<Location> it = griddedSurface.getLocationsIterator();
		System.out.println("*******Evenly Gridded Surface************");
		while(it.hasNext()){
			Location loc = (Location)it.next();
			System.out.println(loc.getLatitude()+","+loc.getLongitude()+","+loc.getDepth());
		}

	}

	@Override
	protected AbstractEvenlyGriddedSurface getNewInstance() {
		FrankelGriddedSurface surf = new FrankelGriddedSurface();
		surf.setNumRowsAndNumCols(numRows, numCols);
		surf.gridSpacingDown = gridSpacingDown;
		surf.gridSpacingAlong = gridSpacingAlong;
		surf.sameGridSpacing = sameGridSpacing;
		surf.faultTrace = faultTrace;
		surf.aveDip = aveDip;
		surf.lowerSeismogenicDepth = lowerSeismogenicDepth;
		surf.name = name;
		surf.upperSeismogenicDepth = upperSeismogenicDepth;
		surf.sameGridSpacing = sameGridSpacing;
		return surf;
	}

}
