package org.opensha.commons.gui.plot.jfreechart.xyzPlot;

import java.awt.BasicStroke;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.geom.Rectangle2D;
import java.io.Serializable;

import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.event.RendererChangeEvent;
import org.jfree.chart.plot.CrosshairState;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.LookupPaintScale;
import org.jfree.chart.renderer.PaintScale;
import org.jfree.chart.renderer.xy.AbstractXYItemRenderer;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYItemRendererState;
import org.jfree.chart.ui.RectangleAnchor;
import org.jfree.chart.util.Args;
import org.jfree.chart.util.PublicCloneable;
import org.jfree.data.Range;
import org.jfree.data.general.DatasetUtils;
import org.jfree.data.xy.IntervalXYDataset;
import org.jfree.data.xy.XYDataset;
import org.jfree.data.xy.XYZDataset;

import com.google.common.base.Preconditions;

/**
 * A renderer that represents data from an {@link XYZDataset} by drawing a
 * color block at each (x, y) point, where the color is a function of the
 * z-value from the dataset.  The example shown here is generated by the
 * {@code XYBlockChartDemo1.java} program included in the JFreeChart
 * demo collection:
 * <br><br>
 * <img src="doc-files/XYBlockRendererSample.png" alt="XYBlockRendererSample.png">
 */
public class XYIntervalBlockRenderer extends AbstractXYItemRenderer
implements XYItemRenderer, Cloneable, PublicCloneable, Serializable {

	/** The paint scale. */
	private PaintScale paintScale;

	/**
	 * Creates a new {@code XYBlockRenderer} instance with default
	 * attributes.
	 */
	public XYIntervalBlockRenderer() {
		this.paintScale = new LookupPaintScale();
	}

	/**
	 * Returns the paint scale used by the renderer.
	 *
	 * @return The paint scale (never {@code null}).
	 *
	 * @see #setPaintScale(PaintScale)
	 */
	public PaintScale getPaintScale() {
		return this.paintScale;
	}

	/**
	 * Sets the paint scale used by the renderer and sends a
	 * {@link RendererChangeEvent} to all registered listeners.
	 *
	 * @param scale  the scale ({@code null} not permitted).
	 *
	 * @see #getPaintScale()
	 */
	public void setPaintScale(PaintScale scale) {
		Args.nullNotPermitted(scale, "scale");
		this.paintScale = scale;
		fireChangeEvent();
	}

	/**
	 * Returns the lower and upper bounds (range) of the x-values in the
	 * specified dataset.
	 *
	 * @param dataset  the dataset ({@code null} permitted).
	 *
	 * @return The range ({@code null} if the dataset is {@code null}
	 *         or empty).
	 *
	 * @see #findRangeBounds(XYDataset)
	 */
	@Override
	public Range findDomainBounds(XYDataset dataset) {
		if (dataset == null) {
			return null;
		}
		// this handles interval bounds already
		return DatasetUtils.findDomainBounds(dataset, false);
	}

	/**
	 * Returns the range of values the renderer requires to display all the
	 * items from the specified dataset.
	 *
	 * @param dataset  the dataset ({@code null} permitted).
	 *
	 * @return The range ({@code null} if the dataset is {@code null}
	 *         or empty).
	 *
	 * @see #findDomainBounds(XYDataset)
	 */
	@Override
	public Range findRangeBounds(XYDataset dataset) {
		if (dataset == null) {
			return null;
		}
		// this handles interval bounds already
		return DatasetUtils.findRangeBounds(dataset, false);
	}

	/**
	 * Draws the block representing the specified item.
	 *
	 * @param g2  the graphics device.
	 * @param state  the state.
	 * @param dataArea  the data area.
	 * @param info  the plot rendering info.
	 * @param plot  the plot.
	 * @param domainAxis  the x-axis.
	 * @param rangeAxis  the y-axis.
	 * @param dataset  the dataset.
	 * @param series  the series index.
	 * @param item  the item index.
	 * @param crosshairState  the crosshair state.
	 * @param pass  the pass index.
	 */
	@Override
	public void drawItem(Graphics2D g2, XYItemRendererState state,
			Rectangle2D dataArea, PlotRenderingInfo info, XYPlot plot,
			ValueAxis domainAxis, ValueAxis rangeAxis, XYDataset dataset,
			int series, int item, CrosshairState crosshairState, int pass) {
		Preconditions.checkState(dataset instanceof IntervalXYDataset, "Must be an IntervalXYDataset");
		IntervalXYDataset interval = (IntervalXYDataset)dataset;

		double x = dataset.getXValue(series, item);
		double y = dataset.getYValue(series, item);
		double z = 0.0;
		if (dataset instanceof XYZDataset) {
			z = ((XYZDataset) dataset).getZValue(series, item);
		}

		Paint p = this.paintScale.getPaint(z);
		double xx0 = domainAxis.valueToJava2D(interval.getStartXValue(series, item),
				dataArea, plot.getDomainAxisEdge());
		double yy0 = rangeAxis.valueToJava2D(interval.getStartYValue(series, item),
				dataArea, plot.getRangeAxisEdge());
		double xx1 = domainAxis.valueToJava2D(interval.getEndXValue(series, item),
				dataArea, plot.getDomainAxisEdge());
		double yy1 = rangeAxis.valueToJava2D(interval.getEndYValue(series, item),
				dataArea, plot.getRangeAxisEdge());
		Rectangle2D block;
		PlotOrientation orientation = plot.getOrientation();
		if (orientation.equals(PlotOrientation.HORIZONTAL)) {
			block = new Rectangle2D.Double(Math.min(yy0, yy1),
					Math.min(xx0, xx1), Math.abs(yy1 - yy0),
					Math.abs(xx0 - xx1));
		}
		else {
			block = new Rectangle2D.Double(Math.min(xx0, xx1),
					Math.min(yy0, yy1), Math.abs(xx1 - xx0),
					Math.abs(yy1 - yy0));
		}
		g2.setPaint(p);
		g2.fill(block);
		g2.setStroke(new BasicStroke(1.0f));
		g2.draw(block);

		if (isItemLabelVisible(series, item)) {
			drawItemLabel(g2, orientation, dataset, series, item, 
					block.getCenterX(), block.getCenterY(), y < 0.0);
		}

		int datasetIndex = plot.indexOf(dataset);
		double transX = domainAxis.valueToJava2D(x, dataArea,
				plot.getDomainAxisEdge());
		double transY = rangeAxis.valueToJava2D(y, dataArea,
				plot.getRangeAxisEdge());        
		updateCrosshairValues(crosshairState, x, y, datasetIndex,
				transX, transY, orientation);

		EntityCollection entities = state.getEntityCollection();
		if (entities != null) {
			addEntity(entities, block, dataset, series, item, 
					block.getCenterX(), block.getCenterY());
		}

	}

	/**
	 * Tests this {@code XYBlockRenderer} for equality with an arbitrary
	 * object.  This method returns {@code true} if and only if:
	 * <ul>
	 * <li>{@code obj} is an instance of {@code XYBlockRenderer} (not
	 *     {@code null});</li>
	 * <li>{@code obj} has the same field values as this
	 *     {@code XYBlockRenderer};</li>
	 * </ul>
	 *
	 * @param obj  the object ({@code null} permitted).
	 *
	 * @return A boolean.
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj == this) {
			return true;
		}
		if (!(obj instanceof XYIntervalBlockRenderer)) {
			return false;
		}
		XYIntervalBlockRenderer that = (XYIntervalBlockRenderer) obj;
		if (!this.paintScale.equals(that.paintScale)) {
			return false;
		}
		return super.equals(obj);
	}

	/**
	 * Returns a clone of this renderer.
	 *
	 * @return A clone of this renderer.
	 *
	 * @throws CloneNotSupportedException if there is a problem creating the
	 *     clone.
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		XYIntervalBlockRenderer clone = (XYIntervalBlockRenderer) super.clone();
		if (this.paintScale instanceof PublicCloneable) {
			PublicCloneable pc = (PublicCloneable) this.paintScale;
			clone.paintScale = (PaintScale) pc.clone();
		}
		return clone;
	}

}