package org.opensha.commons.eq.cat.db;

import org.dom4j.Element;
import org.dom4j.tree.DefaultElement;


/**
 * Wrapper class for an STP server.
 * 
 * @author Peter Powers
 * @version $Id: STP_Server.java 7478 2011-02-15 04:56:25Z pmpowers $
 */
public class STP_Server {
    
    /** XML element names */
    private static final String XML_ELEM_SERVER = "server";
    private static final String XML_ELEM_ADDRESS = "address";
    private static final String XML_ELEM_PASSWORD= "password";
    private static final String XML_ELEM_PORT= "port";
    private static final String XML_ELEM_TIMEOUT = "timeout";
    private static final String XML_ELEM_MAX_EVENTS = "maxEvents";
    
    /** Default timeout (1 min). */
    public static final int DEFAULT_TIMEOUT = 60000;
    
    /** Default maximum number of event server will return (1000). */
    public static final int DEFAULT_MAX_EVENTS = 1000; // default: 1000
    
    private String address;
    private String password;
    private int port;
    private int timeout = DEFAULT_TIMEOUT;
    private int maxEvents = DEFAULT_MAX_EVENTS;
    
    /**
     * Constructs an new, empty STP server configuration.
     */
    public STP_Server() {
    }
    
    /**
     * Constructs a new STP server configuration with a 
     * <code>DEFAULT_TIMEOUT</code> and that will return 
     * <code>DEFAULT_MAX_EVENTS</code> events.
     * 
     * @param address of server
     * @param password for server
     * @param port to commumicate on
     */
    public STP_Server(
            String address,
            String password,
            int port) {
        this.address = address;
        this.password = password;
        this.port = port;
    }

    /**
     * Constructs a new STP server configuration.
     * 
     * @param address of server
     * @param password for server
     * @param port to commumicate on
     * @param timeout of server
     * @param maxEvents to return for any request
     */
    public STP_Server(
            String address,
            String password,
            int port,
            int timeout,
            int maxEvents) {
        
        this.address = address;
        this.password = password;
        this.port = port;
        this.timeout = timeout;
        this.maxEvents = maxEvents;
    }

    /**
     * Constructs a new STP server configuration using the supplied 
     * configuration XML (should be generated by <code>getServerConfig()</code>.
     * No validation is performed on XML so constructor
     * may throw an exception or produce in an empty server object.
     * 
     * @param config to initialize server with
     */
    public STP_Server(Element config) {
        address = config.elementText(XML_ELEM_ADDRESS);
        password = config.elementText(XML_ELEM_PASSWORD);
        port = Integer.parseInt(config.elementText(XML_ELEM_PORT));
        timeout = Integer.parseInt(config.elementText(XML_ELEM_TIMEOUT));
        maxEvents = Integer.parseInt(config.elementText(XML_ELEM_MAX_EVENTS));
    }
    
    /**
     * Returns the server configuration data as an XML element.
     * @return the server config
     */
    public Element getServerConfig() {
        Element server = new DefaultElement(XML_ELEM_SERVER);
        Element e;
        e = new DefaultElement(XML_ELEM_ADDRESS);
        server.add(e.addText(address));
        e = new DefaultElement(XML_ELEM_PASSWORD);
        server.add(e.addText(password));
        e = new DefaultElement(XML_ELEM_PORT);
        server.add(e.addText(String.valueOf(port)));
        e = new DefaultElement(XML_ELEM_TIMEOUT);
        server.add(e.addText(String.valueOf(timeout)));
        e = new DefaultElement(XML_ELEM_MAX_EVENTS);
        server.add(e.addText(String.valueOf(maxEvents)));
        return server;
    }
    
    /**
     * Returns the server address.
     * @return the address
     */
    public String getAddress() {
        return address;
    }

    /**
     * Sets the server address.
     * @param address to set
     */
    public void setAddress(String address) {
        this.address = address;
    }

    /**
     * Returns the server password.
     * @return the password
     */
    public String getPassword() {
        return password;
    }

    /**
     * Sets the server password.
     * @param password to set
     */
    public void setPassword(String password) {
        this.password = password;
    }

    /**
     * Returns the server timeout.
     * @return the timeout
     */
    public int getTimeout() {
        return timeout;
    }

    /**
     * Sets the server timeout in milliseconds.
     * @param timeout the timeout to set
     */
    public void setTimeout(int timeout) {
        this.timeout = timeout;
    }

    /**
     * Returns the server port.
     * @return the port
     */
    public int getPort() {
        return port;
    }

    /**
     * Sets the server communication port.
     * @param port to set
     */
    public void setPort(int port) {
        this.port = port;
    }

    /**
     * Returns the maximum number of events to return for a request.
     * @return the maximum number of events the server will return
     */
    public int getMaxEvents() {
        return maxEvents;
    }

    /**
     * Sets the maximum number of events the server will return.
     * @param maxEvents the number to set
     */
    public void setMaxEvents(int maxEvents) {
        this.maxEvents = maxEvents;
    }

}
